from PyQt5.QtWidgets import (QDialog, QVBoxLayout, QHBoxLayout, QFormLayout, 
                           QLineEdit, QPushButton, QLabel, QMessageBox, QCheckBox,
                           QWidget)
from PyQt5.QtCore import Qt
import json
import os
import ftplib

class FTPSettingsDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle('FTP设置')
        self.setFixedSize(380, 300)
        self.init_ui()
        self.load_settings()

    def init_ui(self):
        layout = QVBoxLayout(self)
        layout.setContentsMargins(20, 15, 20, 15)
        layout.setSpacing(15)

        # 启用FTP复选框
        self.enable_ftp = QCheckBox('启用FTP备份')
        self.enable_ftp.setStyleSheet("""
            QCheckBox {
                font-size: 13px;
                padding: 5px;
            }
            QCheckBox::indicator {
                width: 16px;
                height: 16px;
            }
            QCheckBox::indicator:unchecked {
                border: 1px solid #999;
                background: white;
            }
            QCheckBox::indicator:checked {
                border: 1px solid #666;
                background: #666;
            }
        """)
        layout.addWidget(self.enable_ftp)

        # 创建表单
        form = QFormLayout()
        form.setSpacing(12)
        form.setLabelAlignment(Qt.AlignRight)

        # 创建输入框
        self.ftp_host = QLineEdit()
        self.ftp_port = QLineEdit('21')
        self.ftp_username = QLineEdit()
        self.ftp_password = QLineEdit()
        self.ftp_password.setEchoMode(QLineEdit.Password)

        # 输入框样式
        input_style = """
            QLineEdit {
                padding: 6px 10px;
                border: 1px solid #ddd;
                border-radius: 3px;
                background: white;
                selection-background-color: #e8f0fe;
                font-size: 13px;
                min-height: 20px;
            }
            QLineEdit:focus {
                border-color: #666;
            }
            QLineEdit:disabled {
                background: #f5f5f5;
                color: #999;
            }
        """

        # 标签样式
        label_style = """
            QLabel {
                color: #333;
                font-size: 13px;
                padding-right: 10px;
            }
        """

        # 设置样式
        for widget in [self.ftp_host, self.ftp_port, 
                      self.ftp_username, self.ftp_password]:
            widget.setStyleSheet(input_style)

        # 添加表单项
        labels = ['FTP服务器:', '端口:', '用户名:', '密码:']
        widgets = [self.ftp_host, self.ftp_port, 
                  self.ftp_username, self.ftp_password]

        for label_text, widget in zip(labels, widgets):
            label = QLabel(label_text)
            label.setStyleSheet(label_style)
            form.addRow(label, widget)

        layout.addLayout(form)

        # 按钮容器
        btn_container = QWidget()
        btn_layout = QHBoxLayout(btn_container)
        btn_layout.setContentsMargins(0, 10, 0, 0)
        btn_layout.setSpacing(10)

        # 按钮样式
        btn_base_style = """
            QPushButton {
                padding: 5px 12px;
                border-radius: 3px;
                font-size: 13px;
                min-width: 70px;
            }
        """

        # 测试连接按钮
        test_btn = QPushButton('测试连接')
        test_btn.setStyleSheet(btn_base_style + """
            QPushButton {
                color: #333;
                border: 1px solid #ddd;
                background: #f8f9fa;
            }
            QPushButton:hover {
                background-color: #f1f3f4;
            }
        """)

        # 保存按钮
        save_btn = QPushButton('保存')
        save_btn.setStyleSheet(btn_base_style + """
            QPushButton {
                color: white;
                border: none;
                background: #666;
            }
            QPushButton:hover {
                background-color: #555;
            }
        """)

        # 取消按钮
        cancel_btn = QPushButton('取消')
        cancel_btn.setStyleSheet(btn_base_style + """
            QPushButton {
                color: #333;
                border: 1px solid #ddd;
                background: #f8f9fa;
            }
            QPushButton:hover {
                background-color: #f1f3f4;
            }
        """)

        # 添加按钮
        test_btn.clicked.connect(self.test_connection)
        save_btn.clicked.connect(self.save_settings)
        cancel_btn.clicked.connect(self.reject)

        btn_layout.addWidget(test_btn)
        btn_layout.addStretch()
        btn_layout.addWidget(save_btn)
        btn_layout.addWidget(cancel_btn)

        layout.addWidget(btn_container)

        # 连接复选框状态变化信号
        self.enable_ftp.stateChanged.connect(self.toggle_inputs)
        # 初始状态设置
        self.toggle_inputs(self.enable_ftp.isChecked())

    def toggle_inputs(self, state):
        """启用/禁用输入框"""
        enabled = bool(state)
        for widget in [self.ftp_host, self.ftp_port, 
                      self.ftp_username, self.ftp_password]:
            widget.setEnabled(enabled)

    def test_connection(self):
        """测试FTP连接"""
        if not self.enable_ftp.isChecked():
            QMessageBox.warning(self, '警告', '请先启用FTP备份')
            return
            
        try:
            host = self.ftp_host.text().strip()
            port = int(self.ftp_port.text().strip())
            username = self.ftp_username.text().strip()
            password = self.ftp_password.text().strip()
            
            if not all([host, port, username, password]):
                QMessageBox.warning(self, '警告', '请填写所有字段')
                return
                
            # 测试FTP连接
            ftp = ftplib.FTP()
            ftp.connect(host=host, port=port)
            ftp.login(user=username, passwd=password)
            ftp.quit()
            
            QMessageBox.information(self, '成功', 'FTP服务器连接测试成功!')
            
        except Exception as e:
            QMessageBox.critical(self, '错误', f'连接失败: {str(e)}')

    def load_settings(self):
        """从配置文件加载FTP设置"""
        try:
            settings_path = os.path.join(os.path.expanduser('~'), '.xiaobaissh', 'ftp_settings.json')
            if os.path.exists(settings_path):
                with open(settings_path, 'r') as f:
                    settings = json.load(f)
                    self.enable_ftp.setChecked(settings.get('enabled', False))
                    self.ftp_host.setText(settings.get('host', ''))
                    self.ftp_port.setText(str(settings.get('port', '21')))
                    self.ftp_username.setText(settings.get('username', ''))
                    self.ftp_password.setText(settings.get('password', ''))
                    
            # 初始状态设置输入框启用/禁用
            self.toggle_inputs(self.enable_ftp.isChecked())
            
        except Exception as e:
            print(f"加载FTP设置失败: {e}")

    def save_settings(self):
        """保存FTP设置到配置文件"""
        try:
            if self.enable_ftp.isChecked():
                # 验证输入
                host = self.ftp_host.text().strip()
                port = self.ftp_port.text().strip()
                username = self.ftp_username.text().strip()
                password = self.ftp_password.text().strip()
                
                if not all([host, port, username, password]):
                    QMessageBox.warning(self, '警告', '请填写所有字段')
                    return
                    
                try:
                    port = int(port)
                    if port <= 0 or port > 65535:
                        raise ValueError()
                except ValueError:
                    QMessageBox.warning(self, '错误', '端口号必须是1-65535之间的数字')
                    return
            
            settings = {
                'enabled': self.enable_ftp.isChecked(),
                'host': self.ftp_host.text().strip(),
                'port': self.ftp_port.text().strip(),
                'username': self.ftp_username.text().strip(),
                'password': self.ftp_password.text().strip()
            }
            
            settings_dir = os.path.join(os.path.expanduser('~'), '.xiaobaissh')
            if not os.path.exists(settings_dir):
                os.makedirs(settings_dir)
                
            settings_path = os.path.join(settings_dir, 'ftp_settings.json')
            with open(settings_path, 'w') as f:
                json.dump(settings, f)
            
            QMessageBox.information(self, '成功', 'FTP设置已保存')
            self.accept()
            
        except Exception as e:
            QMessageBox.critical(self, '错误', f'保存FTP设置失败: {str(e)}') 