import json
import os
import sys
import sqlite3
from datetime import datetime
import requests
import urllib.request
import urllib.error
import ssl
import base64

class ScriptManager:
    def __init__(self, db_path="scripts.db"):
        # 在打包环境下，使用用户可写目录保存数据库，避免/Applications内不可写
        if getattr(sys, 'frozen', False):
            app_support = os.path.join(os.path.expanduser('~'), 'Library', 'Application Support', '小白SSH工具')
            try:
                os.makedirs(app_support, exist_ok=True)
            except Exception:
                # 兜底使用用户主目录
                app_support = os.path.expanduser('~')
            self.db_path = os.path.join(app_support, 'scripts.db')
        else:
            # 开发环境使用项目目录下的数据库
            base_dir = os.path.dirname(os.path.abspath(__file__))
            self.db_path = os.path.join(base_dir, db_path)
        self.init_database()
    
    def init_database(self):
        """初始化数据库"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        # 创建脚本表
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS scripts (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                command TEXT NOT NULL,
                script_name TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ''')
        
        # 创建更新记录表
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS update_log (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                update_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                status TEXT,
                message TEXT,
                record_count INTEGER
            )
        ''')
        
        conn.commit()
        conn.close()
    
    def update_from_remote(self):
        """从远程服务器更新脚本数据库"""
        remote_url = 'https://api.irqm.com/dxssd.1dudcl/script/jiaoben.txt'
        username = "ammkiss"
        password = "a1b2c3d4"
        
        try:
            # 清空现有数据
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            cursor.execute('DELETE FROM scripts')
            
            # 尝试从远程获取数据
            success = False
            record_count = 0
            
            # 方法1: 尝试使用requests库
            try:
                response = requests.get(remote_url, timeout=10, verify=False, 
                                     auth=(username, password))
                if response.status_code == 200:
                    content = response.text
                    lines = content.splitlines()
                    
                    for line in lines:
                        if '=' in line:
                            script_name, script_cmd = line.split('=', 1)
                            script_name = script_name.strip()
                            script_cmd = script_cmd.strip()
                            
                            cursor.execute('''
                                INSERT INTO scripts (command, script_name) 
                                VALUES (?, ?)
                            ''', (script_cmd, script_name))
                            record_count += 1
                    
                    success = True
                    
            except ImportError:
                print("requests库不可用，尝试urllib")
            except Exception as e:
                print(f"requests更新失败: {e}")
            
            # 方法2: 如果requests失败，尝试urllib
            if not success:
                try:
                    # 创建Basic Auth头
                    auth_string = f"{username}:{password}"
                    auth_bytes = auth_string.encode('utf-8')
                    auth_b64 = base64.b64encode(auth_bytes).decode('utf-8')
                    
                    # 创建SSL上下文，跳过证书验证
                    ssl_context = ssl.create_default_context()
                    ssl_context.check_hostname = False
                    ssl_context.verify_mode = ssl.CERT_NONE
                    
                    # 创建请求
                    req = urllib.request.Request(remote_url)
                    req.add_header('Authorization', f'Basic {auth_b64}')
                    
                    # 从远程获取
                    with urllib.request.urlopen(req, timeout=10, context=ssl_context) as response:
                        content = response.read().decode('utf-8')
                        lines = content.splitlines()
                        
                        for line in lines:
                            if '=' in line:
                                script_name, script_cmd = line.split('=', 1)
                                script_name = script_name.strip()
                                script_cmd = script_cmd.strip()
                                
                                cursor.execute('''
                                    INSERT INTO scripts (command, script_name) 
                                    VALUES (?, ?)
                                ''', (script_cmd, script_name))
                                record_count += 1
                        
                        success = True
                        
                except Exception as e:
                    print(f"urllib更新失败: {e}")
            
            # 记录更新日志
            if success:
                cursor.execute('''
                    INSERT INTO update_log (status, message, record_count) 
                    VALUES (?, ?, ?)
                ''', ('SUCCESS', f'成功更新 {record_count} 条脚本记录', record_count))
                message = f"成功更新 {record_count} 条脚本记录"
            else:
                cursor.execute('''
                    INSERT INTO update_log (status, message, record_count) 
                    VALUES (?, ?, ?)
                ''', ('FAILED', '更新失败，网络连接或认证问题', 0))
                message = "更新失败，网络连接或认证问题"
            
            conn.commit()
            conn.close()
            
            return success, message, record_count
            
        except Exception as e:
            try:
                conn = sqlite3.connect(self.db_path)
                cursor = conn.cursor()
                cursor.execute('''
                    INSERT INTO update_log (status, message, record_count) 
                    VALUES (?, ?, ?)
                ''', ('ERROR', f'更新过程中发生错误: {str(e)}', 0))
                conn.commit()
                conn.close()
            except:
                pass
            
            return False, f"更新过程中发生错误: {str(e)}", 0
    
    def is_script_command(self, cmd):
        """检查命令是否来自脚本数据库"""
        if not cmd:
            return False
        
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.execute('SELECT COUNT(*) FROM scripts WHERE command = ?', (cmd,))
            count = cursor.fetchone()[0]
            
            conn.close()
            return count > 0
            
        except Exception as e:
            print(f"检查脚本命令失败: {e}")
            return False
    
    def get_script_count(self):
        """获取脚本总数"""
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.execute('SELECT COUNT(*) FROM scripts')
            count = cursor.fetchone()[0]
            
            conn.close()
            return count
            
        except Exception as e:
            print(f"获取脚本数量失败: {e}")
            return 0
    
    def get_last_update_info(self):
        """获取最后更新信息"""
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.execute('''
                SELECT update_time, status, message, record_count 
                FROM update_log 
                ORDER BY update_time DESC 
                LIMIT 1
            ''')
            
            result = cursor.fetchone()
            conn.close()
            
            if result:
                return {
                    'update_time': result[0],
                    'status': result[1],
                    'message': result[2],
                    'record_count': result[3]
                }
            else:
                return None
                
        except Exception as e:
            print(f"获取更新信息失败: {e}")
            return None
    
    def clear_database(self):
        """清空数据库"""
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.execute('DELETE FROM scripts')
            cursor.execute('DELETE FROM update_log')
            
            conn.commit()
            conn.close()
            
            return True, "数据库已清空"
            
        except Exception as e:
            return False, f"清空数据库失败: {str(e)}" 