from PyQt5.QtWidgets import QTextEdit
from PyQt5.QtCore import Qt, pyqtSignal
from PyQt5.QtGui import QTextCursor, QTextCharFormat, QTextBlockFormat

class Terminal(QTextEdit):
    command_signal = pyqtSignal(str)  # 添加信号定义

    def __init__(self, parent=None):
        super().__init__(parent)
        self.setReadOnly(False)
        self.setAcceptRichText(True)
        self.setUndoRedoEnabled(True)
        self.setLineWrapMode(QTextEdit.WidgetWidth)
        self.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        
        # 设置文档格式以控制行高
        doc = self.document()
        doc.setDefaultStyleSheet("""
            span { 
                line-height: 120%; 
                white-space: pre;
            }
            pre {
                margin: 0;
                line-height: 120%;
                white-space: pre;
            }
        """)
        
        self.setStyleSheet("""
            QTextEdit {
                background-color: #1e1e1e;
                color: #ffffff;
                font-family: 'Consolas', 'Menlo', 'Monaco', 'Courier New', monospace;
                font-size: 13px;
                padding: 8px;
                border: none;
                selection-background-color: #264f78;
                selection-color: #ffffff;
                line-height: 120%;
            }
            QTextEdit QScrollBar:vertical {
                width: 12px;
            }
            QTextEdit QScrollBar:horizontal {
                height: 12px;
            }
        """)
        
        # 设置块格式来控制行高
        block_format = QTextBlockFormat()
        block_format.setLineHeight(120, QTextBlockFormat.ProportionalHeight)
        cursor = self.textCursor()
        cursor.setBlockFormat(block_format)
        self.setTextCursor(cursor)
        
        self.command_history = []
        self.history_index = 0
        self.last_block_cursor = 0
        
        # 设置初始提示符
        cursor.insertText("[root@localhost ~]# ")
        self.last_block_cursor = len(self.toPlainText())

    def insertFromMimeData(self, source):
        # 允许粘贴操作
        if source.hasText():
            self.textCursor().insertText(source.text())

    def append_output(self, text):
        """添加输出文本"""
        cursor = self.textCursor()
        cursor.movePosition(QTextCursor.End)
        
        # 处理文本中的换行符，确保正确显示
        lines = text.split('\n')
        for i, line in enumerate(lines):
            # 设置块格式来控制行高
            block_format = QTextBlockFormat()
            block_format.setLineHeight(120, QTextBlockFormat.ProportionalHeight)
            cursor.setBlockFormat(block_format)
            
            if '\x1b[' in line:
                # 处理ANSI转义序列
                processed_text = self._process_ansi_colors(line)
                cursor.insertHtml(processed_text)
            else:
                cursor.insertText(line)
            
            # 在每行之后添加换行符，除非是最后一行且不需要换行
            if i < len(lines) - 1:
                cursor.insertBlock()
        
        # 如果原始文本以换行符结束，添加一个新行
        if text.endswith('\n'):
            cursor.insertBlock()
        
        # 只在合适的时候添加提示符
        if not text.rstrip().endswith("[root@localhost ~]#"):
            cursor.insertText("[root@localhost ~]# ")
        
        self.last_block_cursor = len(self.toPlainText())
        self.setTextCursor(cursor)
        self.ensureCursorVisible()
        
        # 滚动到底部
        scrollbar = self.verticalScrollBar()
        scrollbar.setValue(scrollbar.maximum())

    def keyPressEvent(self, event):
        cursor = self.textCursor()
        
        # 如果光标在提示符之前，移动到末尾
        if cursor.position() < self.last_block_cursor:
            cursor.movePosition(QTextCursor.End)
            self.setTextCursor(cursor)
            
        if event.key() == Qt.Key_Return:
            current_text = self.toPlainText()
            command = current_text[self.last_block_cursor:].strip()
            if command:
                self.command_history.append(command)
                self.history_index = len(self.command_history)
                self.command_signal.emit(command)
            cursor.insertBlock()  # 使用insertBlock代替\n
            cursor.insertText("[root@localhost ~]# ")
            self.last_block_cursor = len(self.toPlainText())
            return
            
        elif event.key() == Qt.Key_Backspace:
            # 防止删除提示符
            if cursor.position() <= self.last_block_cursor:
                return
                
        elif event.key() == Qt.Key_Up:
            if self.command_history and self.history_index > 0:
                self.history_index -= 1
                self.show_history_command()
            return
            
        elif event.key() == Qt.Key_Down:
            if self.history_index < len(self.command_history):
                self.history_index += 1
                self.show_history_command()
            return
            
        super().keyPressEvent(event)

    def show_history_command(self):
        cursor = self.textCursor()
        cursor.setPosition(self.last_block_cursor)
        cursor.movePosition(cursor.End, cursor.KeepAnchor)
        if self.history_index < len(self.command_history):
            cursor.insertText(self.command_history[self.history_index])
        else:
            cursor.insertText('')

    def mousePressEvent(self, event):
        # 允许鼠标选择文本
        super().mousePressEvent(event)

    def mouseDoubleClickEvent(self, event):
        # 允许双击选择单词
        super().mouseDoubleClickEvent(event)

    def mouseMoveEvent(self, event):
        # 允许拖动选择文本
        super().mouseMoveEvent(event)

    def append(self, text):
        """添加文本到终端并处理换行"""
        # 确保换行符正确处理
        cursor = self.textCursor()
        cursor.movePosition(QTextCursor.End)
        
        # 替换特殊的控制字符序列为HTML标签
        # 这有助于保持格式化输出的格式
        formatted_text = text
        
        # 将长行分割成更易读的块
        if len(text) > 80 and '|' in text and '=' in text:
            lines = []
            for line in text.split('\n'):
                # 对于宝塔面板这样的表格式输出，尝试在合适的地方分行
                if len(line) > 80 and ('|' in line or '=' in line):
                    # 在竖线或其他分隔符处分行
                    parts = line.split('|')
                    if len(parts) > 1:
                        # 重新以更整洁的方式组合这些部分
                        formatted_parts = []
                        for part in parts:
                            if part.strip():
                                formatted_parts.append(part.strip())
                        lines.append('\n'.join(formatted_parts))
                    else:
                        lines.append(line)
                else:
                    lines.append(line)
            formatted_text = '\n'.join(lines)
        
        cursor.insertText(formatted_text)
        self.setTextCursor(cursor)
        self.ensureCursorVisible()

    def _replace_sensitive_url(self, text):
        # 直接返回原始文本，不做任何替换
        return text

    def _process_ansi_colors(self, text):
        """改进的ANSI转义序列处理"""
        import re
        
        # ANSI转义序列映射
        color_map = {
            '0': {'style': ''},  # 重置
            '1': {'style': 'font-weight: bold'},
            '30': {'color': '#000000'},
            '31': {'color': '#cd0000'},
            '32': {'color': '#00cd00'},
            '33': {'color': '#cdcd00'},
            '34': {'color': '#0000ee'},
            '35': {'color': '#cd00cd'},
            '36': {'color': '#00cdcd'},
            '37': {'color': '#e5e5e5'},
            '90': {'color': '#7f7f7f'},
            '91': {'color': '#ff0000'},
            '92': {'color': '#00ff00'},
            '93': {'color': '#ffff00'},
            '94': {'color': '#5c5cff'},
            '95': {'color': '#ff00ff'},
            '96': {'color': '#00ffff'},
            '97': {'color': '#ffffff'}
        }

        def process_escape_sequence(match):
            codes = match.group(1).split(';')
            text = match.group(2)
            
            styles = []
            current_color = None
            is_bold = False
            
            for code in codes:
                code = code.strip()
                if code == '0':  # 重置所有属性
                    styles = []
                    current_color = None
                    is_bold = False
                elif code == '1':
                    is_bold = True
                    styles.append('font-weight: bold')
                elif code in color_map:
                    if 'color' in color_map[code]:
                        current_color = color_map[code]['color']
                    elif 'style' in color_map[code]:
                        styles.append(color_map[code]['style'])
            
            if current_color:
                styles.append(f'color: {current_color}')
            
            if styles:
                style_str = '; '.join(styles)
                return f'<span style="{style_str}; white-space: pre">{text}</span>'
            return f'<span style="white-space: pre">{text}</span>'

        # 清理控制序列
        text = re.sub(r'\x1b\[\??(?:\d+)?[ABCDEFGHIJKLMPSTK]', '', text)
        
        # 处理颜色序列
        pattern = r'\x1b\[((?:\d+;)*\d+)m([^\x1b]*)'
        processed_text = re.sub(pattern, process_escape_sequence, text)
        
        return processed_text

    def contextMenuEvent(self, event):
        menu = self.createStandardContextMenu()
        menu.clear()
        
        # 添加复制操作
        copyAction = menu.addAction("复制")
        copyAction.triggered.connect(self.copy)
        
        # 添加全选操作
        selectAllAction = menu.addAction("全选")
        selectAllAction.triggered.connect(self.selectAll)
        
        # 添加清空操作
        clearAction = menu.addAction("清空")
        clearAction.triggered.connect(self.clear)
        
        menu.exec_(event.globalPos())

    def apply_light_theme(self):
        """应用浅色主题到终端"""
        self.setStyleSheet("""
            QTextEdit {
                background-color: white;
                color: #333333;
                font-family: 'Consolas', 'Menlo', 'Monaco', 'Courier New', monospace;
                font-size: 13px;
                padding: 8px;
                border: none;
                selection-background-color: #264f78;
                selection-color: white;
                line-height: 120%;
            }
            QTextEdit QScrollBar:vertical {
                width: 12px;
            }
            QTextEdit QScrollBar:horizontal {
                height: 12px;
            }
        """)
    
    def apply_dark_theme(self):
        """应用深色主题到终端"""
        self.setStyleSheet("""
            QTextEdit {
                background-color: #1e1e1e;
                color: #ffffff;
                font-family: 'Consolas', 'Menlo', 'Monaco', 'Courier New', monospace;
                font-size: 13px;
                padding: 8px;
                border: none;
                selection-background-color: #264f78;
                selection-color: #ffffff;
                line-height: 120%;
            }
            QTextEdit QScrollBar:vertical {
                width: 12px;
            }
            QTextEdit QScrollBar:horizontal {
                height: 12px;
            }
        """)