from PyQt5.QtWidgets import (QDialog, QVBoxLayout, QLabel, QPushButton, 
                            QHBoxLayout, QFrame, QButtonGroup)
from PyQt5.QtCore import Qt, pyqtSignal
from PyQt5.QtGui import QPixmap, QIcon

class ThemeDialog(QDialog):
    """主题选择对话框"""
    themeSelected = pyqtSignal(str)  # 发射选择的主题名称
    
    def __init__(self, parent=None, current_theme="light"):
        super().__init__(parent)
        self.setWindowTitle("选择主题")
        self.setMinimumWidth(400)
        
        # 保存当前主题
        self.current_theme = current_theme
        
        # 设置窗口样式
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)
        
        # 创建界面
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout(self)
        
        # 标题标签
        title_label = QLabel("请选择应用主题", self)
        title_label.setStyleSheet("font-size: 16px; font-weight: bold; margin-bottom: 10px;")
        layout.addWidget(title_label)
        
        # 主题选择区域
        themes_layout = QHBoxLayout()
        
        # 主题选择按钮组
        self.theme_group = QButtonGroup(self)
        
        # 浅色主题框
        light_frame = self._create_theme_option(
            "浅色主题", 
            "资源/light_theme_preview.png",  # 您需要准备主题预览图
            "light",
            """
            QFrame {
                background-color: #f5f5f5;
                border: 2px solid #dddddd;
                border-radius: 8px;
                padding: 8px;
            }
            QFrame[selected=true] {
                border: 2px solid #0078d7;
            }
            QLabel {
                color: #333333;
            }
            """
        )
        themes_layout.addWidget(light_frame)
        
        # 深色主题框
        dark_frame = self._create_theme_option(
            "深色主题", 
            "资源/dark_theme_preview.png",  # 您需要准备主题预览图
            "dark",
            """
            QFrame {
                background-color: #333333;
                border: 2px solid #555555;
                border-radius: 8px;
                padding: 8px;
            }
            QFrame[selected=true] {
                border: 2px solid #0078d7;
            }
            QLabel {
                color: #f0f0f0;
            }
            """
        )
        themes_layout.addWidget(dark_frame)
        
        layout.addLayout(themes_layout)
        
        # 底部按钮区域
        buttons_layout = QHBoxLayout()
        buttons_layout.addStretch()
        
        # 取消按钮
        cancel_btn = QPushButton("取消", self)
        cancel_btn.setObjectName("cancelButton")
        cancel_btn.clicked.connect(self.reject)
        buttons_layout.addWidget(cancel_btn)
        
        # 确认按钮
        ok_btn = QPushButton("确定", self)
        ok_btn.clicked.connect(self._apply_theme)
        buttons_layout.addWidget(ok_btn)
        
        layout.addLayout(buttons_layout)
        
        # 根据当前主题选中相应选项
        if self.current_theme == "dark":
            self.theme_group.button(1).setChecked(True)
            dark_frame.setProperty("selected", True)
            dark_frame.style().polish(dark_frame)
        else:
            self.theme_group.button(0).setChecked(True)
            light_frame.setProperty("selected", True)
            light_frame.style().polish(light_frame)
    
    def _create_theme_option(self, name, img_path, theme_id, style):
        # 创建主题选择框
        frame = QFrame(self)
        frame.setProperty("selected", False)
        frame.setStyleSheet(style)
        
        layout = QVBoxLayout(frame)
        
        # 主题名称
        name_label = QLabel(name, frame)
        name_label.setAlignment(Qt.AlignCenter)
        layout.addWidget(name_label)
        
        # 主题预览图（如果没有图片，可以使用一个纯色区域代替）
        try:
            preview = QLabel(frame)
            pixmap = QPixmap(img_path)
            if not pixmap.isNull():
                preview.setPixmap(pixmap.scaled(150, 100, Qt.KeepAspectRatio, Qt.SmoothTransformation))
                preview.setAlignment(Qt.AlignCenter)
                layout.addWidget(preview)
            else:
                # 创建一个示例预览区域
                example = QFrame(frame)
                example.setMinimumSize(150, 100)
                example.setStyleSheet(f"background-color: {'#333333' if theme_id == 'dark' else '#f5f5f5'};")
                layout.addWidget(example)
        except:
            # 创建一个示例预览区域
            example = QFrame(frame)
            example.setMinimumSize(150, 100)
            example.setStyleSheet(f"background-color: {'#333333' if theme_id == 'dark' else '#f5f5f5'};")
            layout.addWidget(example)
        
        # 选择按钮
        select_btn = QPushButton("选择此主题", frame)
        select_btn.setCheckable(True)
        select_btn.setObjectName(f"theme_{theme_id}")
        # 将按钮添加到按钮组
        self.theme_group.addButton(select_btn, 0 if theme_id == "light" else 1)
        select_btn.toggled.connect(lambda checked, f=frame, t=theme_id: self._on_theme_selected(checked, f, t))
        layout.addWidget(select_btn)
        
        return frame
    
    def _on_theme_selected(self, checked, frame, theme_id):
        """当主题被选中时"""
        if checked:
            # 设置当前主题
            self.current_theme = theme_id
            
            # 更新选中状态的视觉反馈
            for i in range(self.theme_group.buttons().__len__()):
                btn = self.theme_group.button(i)
                if btn:
                    parent_frame = btn.parent()
                    parent_frame.setProperty("selected", btn.isChecked())
                    parent_frame.style().polish(parent_frame)
    
    def _apply_theme(self):
        """应用选中的主题"""
        self.themeSelected.emit(self.current_theme)
        self.accept() 